#include <stdafx.h> //Always need this first in files to compile under MFC.
#include <stdlib.h> //For _lrotr and _lrotl
#include <time.h> //For C library Unix time function
#include "random.h"
#include "math.h" //*for sqrt in rrRandomunitpair
/* This is a set of randomizing functions written by Rudy Rucker,
rucker@mathcs.sjsu.edu.  The randomizing algorithm used is based on
the one-dimensional cellular automaton called Rule 30.  Stephen Wolfram
discovered that this rule makes a good randomizer, see his paper on
the subject in his anthology, Theory and Applications of Cellular
Automata, World Scientific, 1986.  See also Wolfram's Patent #4,691,291
on this rule, and be warned that use of Rule 30 as a randomizer may
be in violation of the patent.  By way of mitigation of this warning,
do note that Rucker has significantly altered the randomizer by adding
the "count" register, and that Rucker wishes to place the so-modified
Wolfram-Rucker randomizer in the public domain.
	This is tweaked to work with 32 bit registers; if you have a differeent
word size you need to change some of the code, for instance the rrRandomreal. */

//-----------------------------------BEGIN 32 BIT CODE------------

static unsigned long shiftregister = 1946; // Rucker's birth year!
static unsigned long count = 0;

unsigned long thirtytwobits(unsigned long seed)
{
/*  In use, we call thirtytwobits(seed) once
 with some nonzero seed, and from then on call it
 with seed value zero to get the next random value.  It also works fine
 if you don't seed it it all, but for short runs you may want to start
 from different seeds. Using the same seed twice produces the same
 sequence of numbers.*/

	register unsigned long l,c,r;

	if (seed)
	{
		shiftregister = seed;
		count = 0;
		return seed;
	}
	l = r = c = shiftregister;
	l = _lrotr(l, 1);/* bit i of l equals bit just left of bit i in c */
	r = _lrotl(r, 1);/* bit i of r euqals bit just right of bit i in c */
	c |= r;
	c ^= l;		/* c = l xor (c or r), aka rule 30, named by wolfram */
	c ^= count;	/* Rucker's trick to  make reaction self-sustaining */
	count++;
	shiftregister = c;
	return c;
}

void rrSeed(unsigned long n)
{
/* This installs a specific seed, so randomizers will repeat. */
	thirtytwobits(n);
}

unsigned int rrRandomize() //We're in 32 bit so int means long.
{
/* This uses the time to install a random seed */
/* Return the seed in case you want it for a regression test.*/
	long timeslot;
	time(&timeslot);	/* unix time function is seconds since 1970*/
	rrSeed( (unsigned long)timeslot );
	return (unsigned int)timeslot;
}

unsigned short rrRandomshort(unsigned short n)
{
/* This returns a random integer from 0 to n - 1. */
/* Bail 0 if n <= 1 to avoid weirdness with modulo operator.*/
	if (n <= 0)
		return 0;
	return ( (unsigned short)((unsigned short)(thirtytwobits(0)) % n) );
}

unsigned long rrRandomlong(unsigned long n)
{
/* This returns a random integer from 0 to n - 1. */
/* Bail 0 if n <= 1 to avoid weirdness with modulo operator.*/
	if (n <= 0)
		return 0;
	return ( (unsigned long)(thirtytwobits(0) % n) );
}

unsigned int rrRandom(unsigned int n)
{  //Here in the __FLAT__ case, an int is 32 bits.
	return (unsigned int)rrRandomlong(n);
}

unsigned char rrRandombyte(void)
{
	return (unsigned char)( thirtytwobits(0) & 0x00FF );
}

double rrRandomreal()
{
/* This returns a random real between 0 and 1. It turns out there is a correlation
between successive random reals gotten this way, so I do a two extra churns of the update
before returning a real */
	thirtytwobits(0);
	thirtytwobits(0);
	return ((double)(thirtytwobits(0))) / 0xFFFFFFFFUL;
/* If you were using this in a sixteen bit machine, you'd do 
	return ((double)(sixteenbits(0))) / 0x10000L; */
}

double rrRandomsign(void)
{
	if (thirtytwobits(0) & 1)
		return 1.0;
	else
		return -1.0;
}
//----------------------------------END 32 BIT CODE------------


//---------------------------------BEGIN DERIVED FUNCTIONS----------

double rrRandomsignreal(void)
{
/* This returns a random real between -1 and 1. */
	return ( -1.0 + 2.0 * rrRandomreal() );
}

double rrRandomreal(double lo, double hi) //A real between lo and hi
{
	return ( lo + (hi-lo) * rrRandomreal() );
}

void rrRandomunitdiskpair(double *x, double *y)
{/* rrRandomizes the pair (x,y) within unit pythagorean distance of (0,0) */
	*x = rrRandomsignreal();
	*y = rrRandomsignreal();
	while ((*x)*(*x) + (*y)*(*y) > 1.0)
	{
		*x = rrRandomsignreal();
		*y = rrRandomsignreal();
	}
}

#ifndef TOO_SMALL
#define TOO_SMALL 0.0000001 //To avoid division by something close to zero
#endif //TOO_SMALL
void rrRandomunitpair(double *x, double *y)
{
	double dist;
	*x = rrRandomsignreal();
	*y = rrRandomsignreal();
	while ((dist = (*x)*(*x) + (*y)*(*y)) > 1.0 || dist < TOO_SMALL)
	{
		*x = rrRandomsignreal();
		*y = rrRandomsignreal();
	}
	dist = 1.0/sqrt(dist); //reciprocal of length
	*x *= dist; //Now make (x,y) a unit vector.
	*y *= dist;
}

